<?php
require_once('modules/Settings/WPPConnect/Config.php');

class WPPConnect_Api {

    /** chiamata HTTP generica */
    private static function call($method, $path, $payload = null, $useAuth = true) {
        $cfg  = Settings_WPPConnect_Config::getConfig();
        $url  = rtrim($cfg['base_url'], '/').'/api/'.ltrim($path, '/');

        $headers = [
            'Accept: application/json',
            'Content-Type: application/json'
        ];

        // Authorization (per quasi tutte le rotte, NON per /generate-token se ancora non hai il token)
        if ($useAuth && !empty($cfg['token'])) {
            $headers[] = 'Authorization: Bearer '.$cfg['token'];
        }

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST  => strtoupper($method),
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_TIMEOUT        => 25,
        ]);
        if ($payload !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        }

        $resp = curl_exec($ch);
        $err  = curl_error($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($err) return ['success'=>false, 'http'=>$http, 'error'=>$err];
        $json = json_decode($resp, true);
        return ['success'=>($http>=200 && $http<300), 'http'=>$http, 'data'=>$json ?: $resp];
    }

    /** Alcune build cambiano i path: prova in sequenza finché una risponde 2xx */
    private static function tryPaths($tries) {
        foreach ($tries as $t) {
            [$m,$p,$body,$auth] = [$t[0], $t[1], $t[2] ?? null, $t[3] ?? true];
            $r = self::call($m,$p,$body,$auth);
            if ($r['success']) return $r;
            $last = $r;
        }
        return $last ?? ['success'=>false,'http'=>0,'error'=>'No endpoint matched'];
    }

    /* ----------------- ENDPOINTS SPECIFICI DELLA TUA BUILD ------------------ */

    /** 1) GENERA TOKEN: /api/{session}/{secretkey}/generate-token  (NO Bearer) */
    public static function generateToken() {
        $cfg = Settings_WPPConnect_Config::getConfig();
        $session   = $cfg['session_name'];
        $secretkey = $cfg['secret_key'];

        // secretkey è nel PATH, non in header.
        $r = self::call('POST', $session.'/'.rawurlencode($secretkey).'/generate-token', null, /*useAuth*/ false);

        // Possibili forme di risposta: adattiamo robustamente
        $token = null;
        if (is_array($r['data'])) {
            $token = $r['data']['session']['token'] ?? $r['data']['token'] ?? $r['data']['data']['token'] ?? null;
        }

        if ($r['success'] && $token) {
            $cfg['token'] = $token;
            Settings_WPPConnect_Config::saveConfig($cfg);
        }
        return $r;
    }

    /** 2) STATUS: /api/{session}/check-connection-session (con Bearer) */
    public static function status() {
        $cfg = Settings_WPPConnect_Config::getConfig();
        return self::call('GET', $cfg['session_name'].'/check-connection-session', null, true);
    }

    /** 3) START SESSION: endpoint varia -> proviamo alcune varianti comuni */
public static function start() {
    $cfg = Settings_WPPConnect_Config::getConfig();
    $s   = $cfg['session_name'];

    // payload con webhook (se c'è) e waitQrCode sempre true
    $payload = [
        'webhook'    => !empty($cfg['webhook']) ? $cfg['webhook'] : '',
        'waitQrCode' => true
    ];

    return self::call('POST', $s.'/start-session', $payload, true);
}


    /** 4) QR CODE: path dipende dalla build */
public static function qrcode() {
    $cfg = Settings_WPPConnect_Config::getConfig();
    $s   = $cfg['session_name'];

    // URL dell’endpoint che nella tua build rende l’immagine
    $url = rtrim($cfg['base_url'], '/').'/api/'.$s.'/qrcode-session';

    $headers = ['Accept: image/png'];
    if (!empty($cfg['token'])) {
        $headers[] = 'Authorization: Bearer '.$cfg['token'];
    }

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER     => $headers,
        CURLOPT_TIMEOUT        => 25,
    ]);
    $png  = curl_exec($ch);
    $err  = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($err || $http >= 400 || !$png) {
        return ['success'=>false, 'http'=>$http, 'error'=>$err ?: 'HTTP '.$http.' no content'];
    }

    // Converto a data URI base64 così JS può metterlo direttamente in <img>
    $dataUri = 'data:image/png;base64,'.base64_encode($png);
    return ['success'=>true, 'http'=>$http, 'data'=>['qrcode'=>$dataUri]];
}


    /** 5) LOGOUT */
    public static function logout() {
        $cfg = Settings_WPPConnect_Config::getConfig();
        $s   = $cfg['session_name'];
        return self::tryPaths([
            ['POST', $s.'/logout-session', null, true],
            ['GET',  $s.'/logout-session', null, true],
            ['POST', $s.'/logout',         null, true],
        ]);
    }
}
